<?php
declare(strict_types=1);

session_start();

/* -----------------------------
   Load .env (tiny loader)
------------------------------ */
function st_env_load(string $path): void {
  if (!is_file($path)) return;
  $lines = file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
  foreach ($lines as $line) {
    $line = trim($line);
    if ($line === '' || str_starts_with($line, '#')) continue;
    if (!str_contains($line, '=')) continue;
    [$k,$v] = explode('=', $line, 2);
    $k = trim($k);
    $v = trim($v);
    $v = trim($v, ""'");
    $_ENV[$k] = $v;
  }
}
st_env_load(__DIR__ . '/../.env');

date_default_timezone_set('Africa/Nairobi');

function env(string $k, ?string $default=null): ?string {
  return $_ENV[$k] ?? getenv($k) ?: $default;
}

function app_name(): string { return (string)env('APP_NAME','ShuleTrack'); }

/* -----------------------------
   Security helpers
------------------------------ */
function csrf_token(): string {
  if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(32));
  return $_SESSION['csrf'];
}
function csrf_check(): void {
  if ($_SERVER['REQUEST_METHOD'] !== 'POST') return;
  $t = (string)($_POST['csrf'] ?? '');
  if (!$t || !hash_equals((string)($_SESSION['csrf'] ?? ''), $t)) {
    http_response_code(419);
    exit('CSRF token mismatch.');
  }
}
function esc($v): string { return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }
function money($n): string { return number_format((float)$n, 2); }

/* -----------------------------
   Database (PDO)
------------------------------ */
function db(): PDO {
  static $pdo = null;
  if ($pdo) return $pdo;

  $host = env('DB_HOST','localhost');
  $name = env('DB_NAME','shuletrack');
  $user = env('DB_USER','root');
  $pass = env('DB_PASS','');
  $dsn  = "mysql:host={$host};dbname={$name};charset=utf8mb4";

  $pdo = new PDO($dsn, $user, $pass, [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
  ]);
  return $pdo;
}

/* -----------------------------
   Auth
------------------------------ */
function current_user(): ?array {
  return $_SESSION['user'] ?? null;
}
function require_login(): array {
  $u = current_user();
  if (!$u) { header('Location: login.php'); exit; }
  return $u;
}
function is_role(string $role): bool {
  $u = current_user();
  return $u && ($u['role'] ?? '') === $role;
}
function tenant_id(): int {
  // Single-school MVP: tenant is always 1, but kept for SaaS readiness.
  $u = current_user();
  return (int)($u['tenant_id'] ?? 1);
}

function flash(string $msg, string $type='info'): void {
  $_SESSION['flash'] = ['msg'=>$msg,'type'=>$type];
}
function take_flash(): ?array {
  if (empty($_SESSION['flash'])) return null;
  $f = $_SESSION['flash'];
  unset($_SESSION['flash']);
  return $f;
}
